#!/usr/bin/env python
# coding: utf-8

# In[13]:


### This program calculates total fluoresence of developing cells under individal masks, generated from a bright-field channel

### Usage: Run in Jupyter Notebook

### Author: Davis Laundon, The Marine Biological Association, July 2020 

#------------------------------------------------------------------------------------------------------------#
### Import the necessary packages before running the script
import os 
import glob

import skimage 
import matplotlib.pyplot as plt
import numpy as np
import scipy

from skimage import io
from skimage import morphology
from skimage.morphology import reconstruction
from skimage import filters
from scipy import ndimage as ndi
from skimage import measure, feature, morphology
from scipy.ndimage import distance_transform_edt

#------------------------------------------------------------------------------------------------------------#
### Load up the images as bright-field and fluoresence lists
bf_image_list = np.sort(glob.glob('##########'))
fluo_image_list = np.sort(glob.glob('#########'))

#------------------------------------------------------------------------------------------------------------#
### Set the manual parameters
sporulation_cutoff = 40 ### The frame at which initial sporulation is expected and afterwards small cells to be cleaned out 
minimum_cell_area = 120 ### No objects smaller than this are included
maximum_cell_area = 2000 ### No objects bigger than this are included
cutoff_cell_size = 800 ### Minimum cell area in pixels, if frame is past sporulation_cutoff 
#pixel_size = 0.3025 ### Pixel area size in um2 -only use if not normalising area data

iteration_limit=50 ### Frame to stop the loop at

#------------------------------------------------------------------------------------------------------------#
fluo_results = [] ### Create an empty array to store the fluorescence data in
area_results = [] ### Create an empty array to store the area results in

for i in range(len(bf_image_list)): ### Set the index for the loop
    
    ### Define the bright field and fluoresence categories
    bf_image = io.imread(bf_image_list[i]) 
    fluo_image = io.imread(fluo_image_list[i]) 
    
    ### Invert the image for segmentation
    bf_invert=np.invert(bf_image)
    
    ### Convert images to 64 floats 
    bf_float = skimage.img_as_float64(bf_invert) 
    fluo_float = skimage.img_as_float64(fluo_image) 
    
    ### Background subtract using a dilation subtraction
    seed = np.copy(bf_float) # Set seeds
    seed[1:-1, 1:-1] =  bf_float.min()
    mask =  bf_float
    dilated = reconstruction(seed, mask, method='dilation') ### Create dilated image
    bf_foreground=(bf_float - dilated) ### Generate foreground by subtraction 
    
    bf_blurred = filters.gaussian(bf_foreground, sigma = 2) ### Blur image
    
    bf_edges = filters.sobel(bf_blurred) ### Find edges
    
    ### Threshold out the edges into a binary mask
    threshold = filters.threshold_otsu(bf_edges) 
    binary_cells = bf_edges > threshold
    
    closed_cells = morphology.binary_closing(binary_cells) ### Close the edges
    
    dilated_cells = morphology.binary_dilation(closed_cells) ### Dilate the cells
    
    filled_cells = ndi.binary_fill_holes(dilated_cells) ### Fill the cells 
    
    ### Separate cells by watershed segementation 
    distance_transformation = distance_transform_edt(filled_cells) ### Make the distance map
    local_maxi = feature.peak_local_max(distance_transformation, indices=False, labels=filled_cells) ### Find distance map peaks
    markers = measure.label(local_maxi)
    separate_cells = morphology.watershed(-distance_transformation, markers, mask=filled_cells) ### Run watershed
     
    ### Run through an if condition to clean out new spores from sporulating cells, if time frame is high enough   
    if i >= sporulation_cutoff:
            cleaned_cells = morphology.remove_small_objects(separate_cells, min_size=cutoff_cell_size)
    else:
        cleaned_cells =  separate_cells
    
    ### Measure the cells
    cell_IDs = skimage.measure.regionprops(cleaned_cells, fluo_float, cache=False) ### Overlay mask and fluoresence
    cell_areas = [r.area for r in cell_IDs] ### Measure cell areas
    cell_intensity = [r.mean_intensity for r in cell_IDs] ### Measure mean intensity
    
    ### Rarely, no cells will be detected and the loop will break. Fill these values as 0 
    if len(cell_areas)==0:
        mean_population_intensity = 0
        total_cell_area = 0
    else:
        total_intensity = np.convolve(cell_areas, cell_intensity) ### Get total intensity by area * mean intensity
        mean_population_intensity = np.mean(total_intensity) ### Create mean population intensity
        total_cell_area = np.sum(cell_areas) ### Create total cell area
    
    ### Add the fluo data to the empty array above 
    fluo_results.append(mean_population_intensity)
    
    ### Add the area data to the empty array above 
    area_results.append(total_cell_area)
    
    ### Generate a visual output of the cells and their ROIs
    fig, ax = plt.subplots(figsize=(15, 15))
    plt.imshow(cleaned_cells, cmap='jet')
    plt.imshow(fluo_image, alpha=0.5, cmap='Reds')
    plt.savefig("#######" %i)
    
    if i==iteration_limit: ### Terminate the loop after a set frame 
        break
        

#------------------------------------------------------------------------------------------------------------#
### Make sure data are numpy arrays 
fluo_array=np.asarray(fluo_results)
area_array=np.asarray(area_results)

### Normalise the fluo data between 0 and 1
norm_fluo_results = (fluo_array-min(fluo_array))/(max(fluo_array)-min(fluo_array))

### Normalise the area data between 0 and 1
norm_area_results = (area_array-min(area_array))/(max(area_array)-min(area_array))

### Scale the area to um2
#area_as_array = np.array(area_results) ### Convert to np.array so can multiple by float later 
#scaled_area=(area_as_array*pixel_size) ### Multiple by previously defined pixel area 

### Combine data for the saving 
combined_data=np.column_stack((norm_fluo_results,norm_area_results))

### Save the fluo_data to csv 
np.savetxt('#######', combined_data, delimiter=',', header='mean_total_intensity, cell_areas')

